/*
 * Copyright (c) 2005, 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package sun.net.httpserver;

import java.io.*;
import java.util.Objects;

/**
 * a class which allows the caller to write up to a defined
 * number of bytes to an underlying stream. The caller *must*
 * write the pre-defined number or else an exception will be thrown
 * and the whole request aborted.
 * normal close() does not close the underlying stream
 */

class FixedLengthOutputStream extends FilterOutputStream
{
    private long remaining;
    private boolean closed = false;
    ExchangeImpl t;

    FixedLengthOutputStream (ExchangeImpl t, OutputStream src, long len) {
        super (src);
        if (len < 0) {
            throw new IllegalArgumentException("Content-Length: " + len);
        }
        this.t = t;
        this.remaining = len;
    }

    public void write (int b) throws IOException {
        if (closed) {
            throw new IOException ("stream closed");
        }
        if (remaining == 0) {
            throw new StreamClosedException();
        }
        out.write(b);
        remaining --;
    }

    public void write (byte[]b, int off, int len) throws IOException {
        Objects.checkFromIndexSize(off, len, b.length);
        if (len == 0) {
            return;
        }
        if (closed) {
            throw new IOException ("stream closed");
        }
        if (len > remaining) {
            // stream is still open, caller can retry
            throw new IOException ("too many bytes to write to stream");
        }
        out.write(b, off, len);
        remaining -= len;
    }

    public void close () throws IOException {
        if (closed) {
            return;
        }
        closed = true;
        if (remaining > 0) {
            t.close();
            throw new IOException ("insufficient bytes written to stream");
        }
        flush();
        LeftOverInputStream is = t.getOriginalInputStream();
        if (!is.isClosed()) {
            try {
                is.close();
            } catch (IOException e) {}
        }
        Event e = new Event.WriteFinished(t);
        t.getHttpContext().getServerImpl().addEvent (e);
    }

    // flush is a pass-through
}
