/*
 * Copyright (c) 2009, 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package com.sun.tools.javap;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.lang.classfile.Attributes;
import java.lang.classfile.CodeModel;
import java.lang.classfile.Instruction;
import java.lang.classfile.Signature;
import java.lang.classfile.attribute.LocalVariableTypeInfo;

/**
 * Annotate instructions with details about local variables.
 *
 *  <p><b>This is NOT part of any supported API.
 *  If you write code that depends on this, you do so at your own risk.
 *  This code and its internal interfaces are subject to change or
 *  deletion without notice.</b>
 */
public class LocalVariableTypeTableWriter extends  InstructionDetailWriter {
    public enum NoteKind {
        START("start") {
            @Override
            public boolean match(LocalVariableTypeInfo entry, int pc) {
                return (pc == entry.startPc());
            }
        },
        END("end") {
            @Override
            public boolean match(LocalVariableTypeInfo entry, int pc) {
                return (pc == entry.startPc() + entry.length());
            }
        };
        NoteKind(String text) {
            this.text = text;
        }
        public abstract boolean match(LocalVariableTypeInfo entry, int pc);
        public final String text;
    }

    static LocalVariableTypeTableWriter instance(Context context) {
        LocalVariableTypeTableWriter instance = context.get(LocalVariableTypeTableWriter.class);
        if (instance == null)
            instance = new LocalVariableTypeTableWriter(context);
        return instance;
    }

    protected LocalVariableTypeTableWriter(Context context) {
        super(context);
        context.put(LocalVariableTypeTableWriter.class, this);
        classWriter = ClassWriter.instance(context);
    }

    public void reset(CodeModel attr) {
        codeAttr = attr;
        pcMap = new HashMap<>();
        var lvt = attr.findAttribute(Attributes.localVariableTypeTable());

        if (lvt.isEmpty())
            return;

        for (var entry : lvt.get().localVariableTypes()) {
            put(entry.startPc(), entry);
            put(entry.startPc() + entry.length(), entry);
        }
    }

    @Override
    public void writeDetails(int pc, Instruction instr) {
        writeLocalVariables(pc, NoteKind.END);
        writeLocalVariables(pc, NoteKind.START);
    }

    @Override
    public void flush(int pc) {
        writeLocalVariables(pc, NoteKind.END);
    }

    public void writeLocalVariables(int pc, NoteKind kind) {
        String indent = space(2); // get from Options?
        var entries = pcMap.get(pc);
        if (entries != null) {
            for (var iter = entries.listIterator(kind == NoteKind.END ? entries.size() : 0);
                    kind == NoteKind.END ? iter.hasPrevious() : iter.hasNext() ; ) {
                var entry = kind == NoteKind.END ? iter.previous() : iter.next();
                if (kind.match(entry, pc)) {
                    print(indent);
                    print(kind.text);
                    print(" generic local ");
                    print(entry.slot());
                    print(" // ");
                    try {
                        print(classWriter.sigPrinter.print(Signature.parseFrom(
                                entry.signature().stringValue())).replace("/", "."));
                    } catch (Exception e) {
                        print(report(e));
                    }
                    print(" ");
                    print(entry.name().stringValue());
                    println();
                }
            }
        }
    }

    private void put(int pc, LocalVariableTypeInfo entry) {
        var list = pcMap.get(pc);
        if (list == null) {
            list = new ArrayList<>();
            pcMap.put(pc, list);
        }
        if (!list.contains(entry))
            list.add(entry);
    }

    private ClassWriter classWriter;
    private CodeModel codeAttr;
    private Map<Integer, List<LocalVariableTypeInfo>> pcMap;
}
